 /*
  * vdev_init_top.c
  * Copyright (C) 2025  Aitor C.Z. <aitor_czr@gnuinos.org>
  * 
  * This program is free software: you can redistribute it and/or modify it
  * under the terms of the GNU General Public License as published by the
  * Free Software Foundation, either version 3 of the License, or
  * (at your option) any later version.
  * 
  * This program is distributed in the hope that it will be useful, but
  * WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  * See the GNU General Public License for more details.
  * 
  * You should have received a copy of the GNU General Public License along
  * with this program.  If not, see <http://www.gnu.org/licenses/>.
  * 
  * See the COPYING file.
  */


#include "libvdev/sglib.h"
#include "libvdev/config.h"
#include "libvdev/util.h"
#include "libvdev/param.h"
#include "libvdev/daemonlet.h"
#include "libvdev/misc.h"
#include "libvdev/sbuf.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <stdbool.h>
#include <mntent.h>
#include <sys/stat.h>

#include <poll.h>
#include <linux/netlink.h>
#include <sys/socket.h>
#include <fcntl.h>
#include <sys/types.h>
#include <signal.h>

const char *progname = "vdev_init_top";

const char *mtab = "/etc/mtab";
const char *mountpoint = "/dev";
static char sysfs_mountpoint[PATH_MAX+1];

typedef char *cstr;
SGLIB_DEFINE_VECTOR_PROTOTYPES(cstr);
SGLIB_DEFINE_VECTOR_FUNCTIONS(cstr);

struct sysfs_scan_context {
    char *uevent_path;
    struct sglib_cstr_vector *frontier;
};

// free a list of cstr vectors
// always succeeds
static int vdev_cstr_vector_free_all(struct sglib_cstr_vector *vec)
{
    // free all strings
    for (int i = 0; i < sglib_cstr_vector_size(vec); i++) {
        if (sglib_cstr_vector_at(vec, i) != NULL) {
            free(sglib_cstr_vector_at(vec, i));
            sglib_cstr_vector_set(vec, NULL, i);
        }
    }
    return 0;
}

// 'buf' must be freed after usage
static void get_sysfs_path(char **buf)
{
    struct mntent *e;
    FILE *fstab = NULL;
   
    fstab = setmntent(mtab, "r");
    if (!fstab) {
        vdev_error("%s: setmntent(): error trying to open /etc/mtab: '%s'\n", 
                   progname, strerror (errno));
        exit(EXIT_FAILURE);
    }
    
    *buf = (char*)malloc(sizeof(char) * 32);
    if (!*buf) {
        vdev_error("%s: Memory allocation failure: '%s'\n", 
                   progname, strerror (errno));
        exit(EXIT_FAILURE);
    }
   
    *buf[0] = '\0';
     
    while ((e = getmntent(fstab))) {
        if (!strcmp(e->mnt_type, "sysfs")) {
            sprintf(*buf, "%s", e->mnt_dir);
            break;
        }
    }
   
    endmntent(fstab);
}

// make the full sysfs path from the dev path, plus an additional path 
// return NULL on OOM
static char *vdev_linux_sysfs_fullpath(char const *sysfs_mountpoint, 
                                       char const *devpath, char const *attr_path)
{
    char *tmp = NULL;
    char *ret = NULL;
   
    tmp = vdev_fullpath(sysfs_mountpoint, devpath, NULL);
    if (!tmp)
        return NULL;
   
    ret = vdev_fullpath(tmp, attr_path, NULL);
    free(tmp);
   
    return ret;
}

// read the kernel-given device subsystem from sysfs 
// return 0 on success, and set *subsystem
// return -ENOMEM on OOM 
// return negative on readlink failure
static int vdev_linux_sysfs_read_subsystem(const char *mountpoint, 
                                           char const *devpath, char **subsystem)
{
    int rc = 0;
    char linkpath[PATH_MAX+1];
    size_t linkpath_len = PATH_MAX;
    char *subsystem_path = NULL;
   
    memset(linkpath, 0, PATH_MAX+1);
   
    subsystem_path = vdev_linux_sysfs_fullpath(mountpoint, devpath, "subsystem");
    if (subsystem_path == NULL)
        return -ENOMEM;
    
    if (access(subsystem_path, F_OK) != 0) {
        /* directory doesn't exist */
        free(subsystem_path);
        return 0;
    }
    
   
    rc = readlink(subsystem_path, linkpath, linkpath_len);
    if (rc < 0) {
        vdev_error("%s: readlink('%s') %s\n", 
                   progname, subsystem_path, strerror(errno));
        free(subsystem_path);
        return -errno;
    }
   
    free(subsystem_path);
   
    *subsystem = vdev_basename(linkpath, NULL);
    if (*subsystem == NULL)
        return -ENOMEM;
   
    return 0;
}

// scan a directory in /sys/devices directory, to find its child directories, that are pushed back to scan_ctx->frontier
// return 0 on success
// return -ENOMEM on OOM
// return -errno on failure to stat
static int scan_device_directory(char const *fp, void *cls)
{
    struct sysfs_scan_context *scan_ctx = (struct sysfs_scan_context *)cls;
   
    struct sglib_cstr_vector *frontier = scan_ctx->frontier;

    int rc = 0;  
    struct stat sb;
    char *fp_base = NULL;
    char *fp_dup = NULL;
   
    fp_base = rindex (fp, '/') + 1;
   
    if (fp_base == NULL)
        return 0;
   
    // skip . and .. 
    if (strcmp(fp_base, ".") == 0 || strcmp(fp_base, "..") == 0)
        return 0;
  
    // add directories
    rc = lstat (fp, &sb);
    if (rc != 0) {
        vdev_error("%s: lstat('%s'): '%s'\n", progname, fp, strerror(errno));
        return -errno;
    }
   
    if (!S_ISDIR(sb.st_mode) && strcmp(fp_base, "uevent") != 0)
        return 0;
   
    fp_dup = vdev_strdup_or_null(fp);
    if (fp_dup == NULL)
        return -ENOMEM;
   
    if (S_ISDIR(sb.st_mode)) {
        rc = sglib_cstr_vector_push_back(frontier, fp_dup);
        if (rc != 0) {
            vdev_error("%s: sglib_cstr_vector_push_back('%s'): '%s'\n", 
                       progname, fp_dup, strerror(errno));
            free (fp_dup);
            return rc;
        }
    /* this is a uevent; this directory is a device */      
    } else {
        scan_ctx->uevent_path = fp_dup;
    }
   
    return 0;
}

/**
 * \brief Variadic function
 */
static int find_devices_at_frontier(char *sysfs_mountpoint, 
                                    const char *device_frontier, 
                                    struct sglib_cstr_vector *uevent_paths)
{
    int rc = 0;
    struct sglib_cstr_vector frontier;
    struct sysfs_scan_context scan_ctx;

    sglib_cstr_vector_init (&frontier);
   
    memset(&scan_ctx, 0, sizeof(struct sysfs_scan_context));
   
    scan_ctx.frontier = &frontier;

    rc = vdev_load_all(device_frontier, scan_device_directory, &scan_ctx);
    if (rc != 0) {
        vdev_error("%s: vdev_load_all('%s'): '%s'\n", 
                   progname, device_frontier, strerror(errno));
        vdev_cstr_vector_free_all(&frontier);
        sglib_cstr_vector_free(&frontier);      
        return rc;
    }
   
    while (1) {
        int len = sglib_cstr_vector_size (&frontier);
        if (len == 0)
            break;
   
        char *dir = sglib_cstr_vector_at (&frontier, len - 1);
        sglib_cstr_vector_set(&frontier, NULL, len - 1);
      
        sglib_cstr_vector_pop_back(&frontier);
      
        // scan for more devices 
        rc = vdev_load_all(dir, scan_device_directory, &scan_ctx);
        if (rc != 0) {
            vdev_error("%s: vdev_load_all('%s'): '%s'\n", 
                       progname, dir, strerror(errno));
            free (dir);
            break;
        }
      
        // is one of them a uevent?
        if (scan_ctx.uevent_path) {
            const char *str;
            bool is_ok = false;
     
            char *uevent_path = vdev_strdup_or_null(scan_ctx.uevent_path + 
                                                    strlen(sysfs_mountpoint));
            if (!uevent_path) {
                free (dir);
                free (scan_ctx.uevent_path);
                scan_ctx.uevent_path = NULL;
                rc = -ENOMEM;
                break;
            }

            rc = sglib_cstr_vector_push_back(uevent_paths, uevent_path);
            if (rc < 0) {
                vdev_error("%s: sglib_cstr_vector_push_back('%s'): '%s'\n", 
                           progname, uevent_path, strerror(errno));
                free(uevent_path);
                free(dir);
                free(scan_ctx.uevent_path);
                scan_ctx.uevent_path = NULL;
                break;
            }
     
            free (scan_ctx.uevent_path);
            scan_ctx.uevent_path = NULL;
        }
      
        free (dir);
    }

    vdev_cstr_vector_free_all(&frontier);
    sglib_cstr_vector_free(&frontier);

    return rc;
}

static int find_devices(char *sysfs_mountpoint, 
                        struct sglib_cstr_vector *uevent_paths)
{
    int rc = 0;
    struct sglib_cstr_vector frontier;
    struct sysfs_scan_context scan_ctx;
   
    memset(&scan_ctx, 0, sizeof(struct sysfs_scan_context));
      
    char *devroot = NULL;

    sglib_cstr_vector_init (&frontier);
   
    scan_ctx.frontier = &frontier;
   
    devroot = vdev_fullpath(sysfs_mountpoint, "/devices", NULL);
    if (devroot == NULL)      
        return -ENOMEM;
    
    rc = vdev_load_all(devroot, scan_device_directory, &scan_ctx);
    if (rc != 0) {
        vdev_error("%s: vdev_load_all('%s'): '%s'\n", 
                   progname, devroot, strerror(errno)); 
        free (devroot);
        vdev_cstr_vector_free_all(&frontier);   
        sglib_cstr_vector_free(&frontier);      
        return rc;
    }
   
    free(scan_ctx.uevent_path);
    scan_ctx.uevent_path = NULL;
 
    for (int i = 0; i < sglib_cstr_vector_size(&frontier); i++)
        find_devices_at_frontier(sysfs_mountpoint, 
                                 sglib_cstr_vector_at(&frontier, i), uevent_paths);
    
    vdev_cstr_vector_free_all(&frontier);
    sglib_cstr_vector_free(&frontier);
    
    if (devroot != NULL) {
        free(devroot);
        devroot = NULL;
    }
   
    return rc;
}

// get a uevent from a uevent file 
// replace newlines with '\0', making the uevent look like it came from the netlink socket
// (i.e. so it can be parsed by vdev_linux_parse_request)
// return 0 on success
// return -ENOMEM on OOM
// return -errno on failure to stat or read
static int vdev_read_uevent(char const *fp_uevent, 
                            char **ret_uevent_buf, size_t *ret_uevent_len)
{
    int rc = 0;
    struct stat sb;
    char *uevent_buf = NULL;
    size_t uevent_buf_len = 0;
    size_t uevent_len = 0;
   
    // get uevent size  
    rc = stat(fp_uevent, &sb);
    if (rc != 0) {
        vdev_error("%s: stat('%s'): '%s'\n", 
                   progname, fp_uevent, strerror (errno));
        return -errno;
    }
    else
        uevent_buf_len = sb.st_size;
   
    // read the uevent as long as it's a regular file
    if (fp_uevent != NULL && (sb.st_mode & S_IFMT) == S_IFREG) {
        uevent_buf = VDEV_CALLOC(char, uevent_buf_len);
        if (uevent_buf == NULL)
            return -ENOMEM;
      
        rc = vdev_read_file(fp_uevent, uevent_buf, uevent_buf_len);
        if (rc != 0) {
            // failed in this 
            vdev_error("%s: read_file('%s'): '%s'\n", 
                       progname, fp_uevent, strerror (errno));
            free(uevent_buf);
        }
        else {
            for (unsigned int i = 0; i < uevent_buf_len; i++) {
                if (uevent_buf[i] == '\n')           
                    uevent_buf[i] = '\0';
            }
     
            // NOTE: the stat size is an upper-bound.  Find the exact number of bytes.
            for (uevent_len = 0; uevent_len < uevent_buf_len;) {
                if (*(uevent_buf + uevent_len) == '\0')
                    break;
                uevent_len += strlen(uevent_buf + uevent_len) + 1;
            }
      
            *ret_uevent_buf = uevent_buf;
            *ret_uevent_len = uevent_len;
        }
    }
   
    return rc;
}

// carry out the modprobe command by sending it to the running daemonlet.
// restart the daemonlet if we need to (e.g. if it hasn't been started, or it died since the last device request).
// return the daemonlet's exit status on success (will be positive if the daemonlet failed in error).
// return -ENOMEM on OOM
// return -EPERM if the daemonlet could not be started, or was not responding and we could not restart it.  A subsequent call probably won't succeed.
// return -EAGAIN if the daemonlet was not responding, but we were able to stop it. A subsequent call might succeed in starting it up and feeding it the request.
static int feed_modprobe_daemonlet(char const *fp_uevent,
                   struct vdev_config *config,
                   struct vdev_daemonlet *daemonlet,
                   const char *name,
                   const char *command,
                   int *error_fd)
{
    int rc = 0;
    struct stat sb;
    char *uevent_buf = NULL;
    size_t uevent_buf_len = 0;
    char *full_devpath = NULL;
    sbuf_t s;     
    char **env = NULL;
    size_t num_env = 2;
    int64_t daemonlet_rc = 0;         
   
    // get uevent    
    sbuf_init(&s);
    sbuf_concat(&s, 2, sysfs_mountpoint, fp_uevent);
    rc = vdev_read_uevent(s.buf, &uevent_buf, &uevent_buf_len);
    if (rc != 0) {
        vdev_error("%s: read_uevent('%s'): '%s'\n", 
                   progname, fp_uevent, strerror(errno));
        return rc;
    }
    sbuf_free(&s);
   
    if (uevent_buf_len == 0) {
        free(uevent_buf);
        return 0;
    }             
     
    env = VDEV_CALLOC(char*, num_env + 1);    
    rc = vdev_make_env_str("VDEV_DAEMONLET", "1", &env[0]);
    if (rc != 0) {
        VDEV_FREE_LIST(env);
        return rc;
    } 
  
    for (unsigned int i = 0; i < uevent_buf_len;) {
        char *tmp = rindex(uevent_buf + i, '=') + 1;
        if (!strncmp(uevent_buf+i, "MODALIAS=", strlen("MODALIAS=")) && strlen(tmp) > 0) {
            char *tmp = rindex(uevent_buf + i, '=') + 1;
            if (strlen(tmp) > 2) {
                int num_attempts = 0;
                rc = vdev_make_env_str("VDEV_OS_MODALIAS", tmp, &env[1]);
                if (rc != 0) {
                    VDEV_FREE_LIST(env);
                    return rc;
                }
                /* try twice, in case we need to stop and start it */
                while (num_attempts < 2) {
                    rc = vdev_daemonlet_send_command(env, 
                                     num_env,
                                     name,
                                     false,
                                     &daemonlet_rc,
                                     daemonlet);
                    if (rc != 0) {
                        vdev_error("vdev_action_daemonlet_send_command('%s') rc = %d\n", 
                                   name, rc);
                        if (rc == -EAGAIN) {
                            rc = vdev_daemonlet_stop(daemonlet, name);
                            if (rc < 0) {
                                vdev_error("vdev_action_daemonlet_stop('%s', PID=%d) rc = %d\n", 
                                           name, daemonlet->pid, rc);
                                rc = -EPERM;
                                break;
                            } else {
                                rc = vdev_daemonlet_start(config,
                                              daemonlet,
                                              command,
                                              name,
                                              false,
                                              error_fd);
                                if (rc < 0) {
                                    vdev_error("vdev_action_daemonlet_start('%s') rc = %d\n",
                                           name, rc);
                                    rc = -EPERM;
                                    break;
                                } else {  
                                    num_attempts++;
                                    continue;
                                }
                            }
                        } else {
                            rc = -EPERM;
                            break;
                        }
                    } else {
                        break;
                    }
                }
            }
        }
        i += strlen(uevent_buf + i) + 1;
    }
    
    VDEV_FREE_LIST(env);

    if (rc == 0) {
        vdev_debug("daemonlet '%s' returned %d\n", name, (int)daemonlet_rc);
        return (int)daemonlet_rc;
    } else {
        return rc;
    }
}

int main(int argc, char **argv)
{
    int rc;
    sbuf_t s;
    char *buf = NULL;
    struct vdev_config *config;
    struct sglib_cstr_vector uevent_paths;
    struct sglib_cstr_vector pci_paths;
    int error_fd = 0;
    bool async = false;
    struct vdev_daemonlet *daemonlet;
    const char *command = "/lib/vdev/modprobe.sh";
    const char *name = "/etc/vdev/actions/002-modalias.act";
    const char *config_file = "/etc/vdev/vdevd.initrd.conf";
    
    FILE *pfin = NULL;
    pid_t pid;
    int wstatus;
    char *cmd = NULL;
   
    if (access(mtab, F_OK) != 0) {
        snprintf (sysfs_mountpoint, PATH_MAX, "/sys"); 
    } else {
        get_sysfs_path (&buf);
        if (buf && buf[0] != '\0') {
            snprintf (sysfs_mountpoint, PATH_MAX, buf);       
            free (buf);   
        } else {
            vdev_error("%s: Cannot get sysfs path\n", progname);
            exit (EXIT_FAILURE);
        }
    }
   
    config = VDEV_CALLOC(struct vdev_config, 1);
    if (config == NULL)
        return -ENOMEM;
   
    rc = vdev_config_init(config);
    if (rc != 0) {
        vdev_error("vdev_config_init rc = %d\n", rc);
        return rc;
    }
    
    rc = vdev_config_load(config_file, config);
    if (rc != 0) {
        vdev_error("vdev_config_load('%s') rc = %d\n", config->config_path, rc);
        return rc;
    }
    
    config->config_path = vdev_strdup_or_null(config_file);
    if (!config->config_path)  // OOM
        return -ENOMEM;
    
    config->mountpoint = vdev_strdup_or_null(mountpoint);
    if (!mountpoint)
        return -ENOMEM;
   
    cmd = VDEV_CALLOC(char, strlen(config->preseed_path) + 4 + 
                            strlen(mountpoint) + 2 + 
                            strlen(config_file) + 1);
    if (!cmd)
        return -ENOMEM;  // OOM
    
    /* 1 or 0 is irrelevant here */
    sprintf(cmd, "%s 1 %s %s", config->preseed_path, mountpoint, config_file);
    pfin = epopen(cmd, &pid);
    if (pfin) { 
           fclose(pfin);
           waitpid(pid, &wstatus, 0);
    }
    free(cmd);
    cmd = NULL;
    
    send_to_background();

    sglib_cstr_vector_init(&uevent_paths);
    rc = find_devices(sysfs_mountpoint, &uevent_paths);
    
    sglib_cstr_vector_init (&pci_paths);
       
    for (unsigned long i = 0; i < sglib_cstr_vector_size(&uevent_paths); i++) {
        sbuf_t aux;
        sbuf_init(&aux);
        sbuf_concat(&aux, 2, sysfs_mountpoint, sglib_cstr_vector_at(&uevent_paths, i));
        if (sysfs_get_parent_with_subsystem(aux.buf, "pci")) {
            char *copy = vdev_strdup_or_null(sglib_cstr_vector_at (&uevent_paths, i));
            if (!copy) {
                rc = -ENOMEM;
                sbuf_free(&aux);
                goto Free_paths;
            } 
            rc = sglib_cstr_vector_push_back(&pci_paths, copy);
            if (rc != 0) {
                vdev_error("%s: sglib_cstr_vector_push_back('%s'): '%s'\n", 
                           progname, copy, strerror(errno));
                free(copy);
                sbuf_free(&aux);
                goto Free_paths;
            }
            sglib_cstr_vector_set(&uevent_paths, NULL, i);
        }
        sbuf_free(&aux);
    }
    vdev_cstr_vector_free_all(&uevent_paths);       
    sglib_cstr_vector_free(&uevent_paths);
   
    // ignore SIGPIPE from daemonlets 
    signal(SIGPIPE, SIG_IGN);

    daemonlet = VDEV_CALLOC(struct vdev_daemonlet, 1);
    if (daemonlet == NULL)
        return -ENOMEM; 
           
    // do we need to start it?
    if (daemonlet->pid <= 0) {
        rc = vdev_daemonlet_start(config,
                      daemonlet,
                      command,
                      name,
                      async,
                      &error_fd);
        if (rc < 0) {
            vdev_error("vdev_action_daemonlet_start('%s') rc = %d\n", name, rc);
            return -EPERM;
        }
    } 
     
    char *p[] = { "usb", "block", "pci", (char*)0 };
    for (unsigned i = 0; i < sizeof(p)/sizeof(char*) - 1; i++) {
        for (unsigned long j = 0; j < sglib_cstr_vector_size(&pci_paths); j++) {
            char *subsystem = NULL;
            char *full_devpath = NULL;
            
            // extract the devpath from the uevent path 
            full_devpath = vdev_dirname(sglib_cstr_vector_at(&pci_paths, j), NULL);
            if (full_devpath == NULL)
                continue;

            rc = vdev_linux_sysfs_read_subsystem(sysfs_mountpoint, 
                                                 full_devpath, &subsystem);
            if (rc == 0) {
                if (subsystem && !strcmp(subsystem, p[i])) {
                    feed_modprobe_daemonlet(sglib_cstr_vector_at(&pci_paths, j),
                                config,
                                daemonlet,
                                name,
                                command,
                                &error_fd);
                    sglib_cstr_vector_set(&pci_paths, NULL, j);
                }
                free(subsystem);
            }
            free(full_devpath);
        }
    }
    goto Free_pci_paths;

Free_paths:
    vdev_cstr_vector_free_all(&uevent_paths);       
    sglib_cstr_vector_free(&uevent_paths);

Free_pci_paths:    
    vdev_cstr_vector_free_all(&pci_paths);
    sglib_cstr_vector_free (&pci_paths);
    
    vdev_daemonlet_stop(daemonlet, name);
    
    if (daemonlet != NULL) {
        free(daemonlet);
        daemonlet = NULL;
    }
    
    if (config != NULL) {
        vdev_config_free(config);
        free(config);
        config = NULL;
    }

    return 0;
}
