#include <stdio.h>
#include <sys/types.h>
#include <wait.h>
#include <errno.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <assert.h>
#include "hopman.h"
#include "watch.h"

// Usage: strncpy_t(str1, sizeof(str1), str2, strlen(str2));
// Copy string "in" with at most "insz" chars to buffer "out", which
// is "outsz" bytes long. The output is always 0-terminated. Unlike
// strncpy(), strncpy_t() does not zero fill remaining space in the
// output buffer:
static
char* strncpy_t(char* out, size_t outsz, const char* in, size_t insz)
{
   assert(outsz > 0);
   while (--outsz > 0 && insz > 0 && *in) { *out++ = *in++; insz--; }
   *out = 0;
   return out;
}

void execute(command_t command, partinfo_t *p)
{
    static int debug = -1;
    pid_t pid;
    FILE *pfin = NULL;
    int wstatus;
    char *line;
    int len;
    char cmdbuf[1024];
    const char *cmdline;
    char cmd[512] = {0};
    char target[1024] = {0};
    FILE *pf = NULL;

    if(debug == -1)
      {
        debug = config_bool(Debug);
        if(debug == -1) debug = 0;
      }
  
    cmdline = config_helper(command, p, cmdbuf, sizeof(cmdbuf));
    if(!cmdline)
      {
        ui_msg("No command helper.");
        return;
      }
    if(debug)
        fprintf(stderr, "External command to be issued: \"%s\"\n", cmdline);
	
    strncpy_t(cmd, sizeof(cmd), "/bin/findmnt -o TARGET -n /dev/", 
                strlen("/bin/findmnt -o TARGET -n /dev/"));
    strcat(cmd, p->name);
    
    pf = popen(cmd, "r");
    if (! pf)
      {
        fprintf(stderr, "%s: popen(): %s", progname, strerror(errno));
        exit(-errno);
      }
    
    if (fgets(target, sizeof(target), pf))
      {
        target[strcspn(target, "\n")] = '\0';
        if (target[0] == '/')
          {
            int rc = chdir(target);
            if (rc == -1)
              {
                fprintf(stderr, _("%s: chdir(%s): %s\n"), progname, target, strerror(errno));
                exit(-errno);
              }
          }  
      }
  
    switch (command)
      {
        case Open_in_File_Manager:
            run_async_function(cmdline, run_file_manager);
            break;
        case Open_in_Terminal:
            run_async_function(cmdline, run_terminal_emulator);
            break;
        case Mount:
            if(!p->nmounts)
                pfin = epopen(cmdline, &pid, "");
            else
                ui_msg("already mounted.");
            break;
        case Unmount:
            if(p->nmounts)
                pfin = epopen(cmdline, &pid, "");
            else
                ui_msg("not mounted.");
            break;
        case Eject:
            pfin = epopen(cmdline, &pid, "");
            break;
        default:
            ui_msg("internal program error: invalid command.");
      }
    
    if(pfin)
      {
        line = fgets(cmdbuf, sizeof(cmdbuf), pfin);
        if(line)
	      {
            len = strlen(cmdbuf);
            if(cmdbuf[len -1] == '\n')
              {
                cmdbuf[len -1] = '\0';
                len --;
              }
            if(len)
              {
                if(debug) fprintf(stderr, "Message: \"cmdbuf\"\n", cmdbuf);
                ui_msg(cmdbuf);
              }
	      }

        fclose(pfin);
        waitpid(pid, &wstatus, 0);
      }
    
    chdir("/");
    pclose(pf);
    mountpoints(); /* always update */  
    return;
}

/* Reap long lasting child processes */
void command_finished(pid_t pid, int wstatus)
{
  char message[80];

  message[0] = '\0';
  
  if(WIFEXITED(wstatus))
    {
      int status = WEXITSTATUS(wstatus);
      switch(status)
	{
	case EXIT_SUCCESS:
	  /* Don't bother ! */
	  break;
	case EXIT_FAILURE:
	  fprintf( stderr, _("%s helper command with pid %u failed.\n"),
		   progname, pid );
	  break;
	default:
	  fprintf(stderr,
		  _("%s helper command with pid %u terminated with status %d.\n"),
		  progname, pid, status );
	}
    }
  else if(WIFSIGNALED(wstatus))
    {
      int signal = WTERMSIG(wstatus);
      fprintf( stderr,
	       _("%s helper command with pid %u terminated by signal %d.\n"),
	       progname, pid, signal );
    }
  else if(WIFSTOPPED(wstatus))
    {
      int signal = WSTOPSIG(wstatus);
      fprintf( stderr, _("%s helper command with pid %u stopped by signal %d\n"),
	       progname, pid, signal );
    }
}
