
/*  view.c
 *
 *  This file handles the GUI. It "owns" the model and cfg.
 *
 *  Copyright (C) 2022 Aitor Cuadrado Zubizarreta <aitor_czr@gnuinos.org>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gtk/gtk.h>

#include <libxfce4util/libxfce4util.h>
#include <libxfce4panel/libxfce4panel.h>
#include <libxfce4ui/libxfce4ui.h>
#include <string.h>
#include <unistd.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <ctype.h>
#include <time.h>
#include <fcntl.h>

#include "hopman.h"
#include "view.h"
#include "gui.h" // set_icon()

#define MAX_LINE_LENGTH 256
#define MAX_SUBMENU_DEPTH 16
#define BUFFER_LEN 2048
#define MAX_HOME_LEN 128

// Global variables
char Line[MAX_LINE_LENGTH], data[BUFFER_LEN];
int depth, lineNum, menuX, menuY;
FILE *pFile;
GtkWidget *menu[MAX_SUBMENU_DEPTH];

// Function prototypes
int ReadLine ();
gboolean Get2Numbers (char *data);
//static void Expand_Tilde (char *txt, int maxLen);

void printstatus(int status, void *arg);
gboolean callback_inotify( GIOChannel *source, GIOCondition condition,
			   gpointer pData );
gboolean callback_signal ( GIOChannel *source, GIOCondition condition,
			   gpointer pData );
gboolean callback_mounts ( GIOChannel *source, GIOCondition condition,
			   gpointer pData );
		       
void callback_hide( GtkWidget *widget, GdkEvent *event, gpointer *pData );
struct initial_position decode_pos(const char *);		       

/* Root of the device files tree: */
const char *watchdir="/dev";

gboolean decorate_window, toggle_hide, keep_position;

/* Type and global variable to handle initial position configuration */
struct initial_position
{
  enum position_style {None, Mouse, Percent} ip_sty;
  int ip_x, ip_y;
};

struct initial_position ipos;

/* UI Helper */
static void pview_open_menu(HopmanView *);
static void pview_update_menu(HopmanView *);
static void pview_destroy_menu(HopmanView *);

/********** Gtk Callbacks **********/

static void
pview_cb_menu_deact(HopmanView *pd, GtkWidget *menu)
{
    /* deactivate button */
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(pd->button), FALSE);
}

/* Button */
static gboolean
pview_cb_button_pressed(HopmanView *pd, GdkEventButton *evt)
{    
    /* (it's the way xfdesktop menu does it...) */
    if((evt->state & GDK_CONTROL_MASK) && !(evt->state & (GDK_MOD1_MASK|GDK_SHIFT_MASK|GDK_MOD4_MASK)))
        return FALSE;

    if(evt->button == 1) {

        pview_open_menu(pd);
        
    } else if(evt->button == 2) {
      
        //
    }

    return FALSE;
}

static void
open_menu_at_pointer (GtkMenu *menu)
{
    GdkDevice    *mouse_device;
    GdkRectangle  rect;
    GdkSeat      *seat;
    GdkWindow    *window;
    gint          x, y;

    window = gdk_display_get_default_group (gdk_display_get_default ());
    seat = gdk_display_get_default_seat (gdk_display_get_default ());
    mouse_device = gdk_seat_get_pointer (seat);
    gdk_window_get_device_position (window,
                                    mouse_device,
                                    &x, &y,
                                    NULL);
    rect.x = x;
    rect.y = y;
    rect.width = gdk_window_get_width (window);
    rect.height = gdk_window_get_height (window);

    gtk_menu_popup_at_rect (menu,
                            window,
                            &rect,
                            GDK_GRAVITY_NORTH_WEST,
                            GDK_GRAVITY_NORTH_WEST,
                            NULL);
}

static void
open_menu_at_widget (GtkMenu   *menu,
                     GtkWidget *widget)
{
    gtk_menu_popup_at_widget (menu,
                              widget,
                              GDK_GRAVITY_SOUTH_WEST,
                              GDK_GRAVITY_NORTH_WEST,
                              NULL);
}

/********** UI Helpers **********/

static void
pview_open_menu_at (HopmanView   *pd,
                    GtkWidget    *button)
{
    /* check if menu is needed, or it needs an update */
    if(pd->menu == NULL)
        pview_update_menu(pd);

    /* toggle the button */
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(pd->button), TRUE);

    /* popup menu */
    DBG("menu: %lu", (gulong)pd->menu);
    if (button == NULL)
        open_menu_at_pointer (GTK_MENU (pd->menu));
    else
        open_menu_at_widget (GTK_MENU (pd->menu), button);
}

static void
pview_open_menu(HopmanView *pd)
{
  if (pd != NULL)
    pview_open_menu_at (pd, pd->button);
}

static void
pview_destroy_menu(HopmanView *view)
{
    if(view->menu != NULL) {
        gtk_menu_shell_deactivate(GTK_MENU_SHELL(view->menu));

        gtk_widget_destroy(view->menu);
        view->menu = NULL;
    }
}

static void
pview_update_menu(HopmanView *pd)
{
	DBG("destroy menu");

    /* destroy the old menu, if it exists */
    pview_destroy_menu(pd);

    DBG("building new menu");

    /* Create a new menu */
    pd->menu = gtk_menu_new();

    /* make sure the menu popups up in right screen */
    gtk_menu_attach_to_widget(GTK_MENU(pd->menu), pd->button, NULL);
    gtk_menu_set_screen(GTK_MENU(pd->menu),
                        gtk_widget_get_screen(pd->button));

    init_list();

    /* connect deactivate signal */
    g_signal_connect_swapped(pd->menu, "deactivate",
                             G_CALLBACK(pview_cb_menu_deact), pd);

    /* Quit hiding the menu */
    gtk_widget_show(pd->menu);

    /* This helps allocate resources beforehand so it'll pop up faster the first time */
    gtk_widget_realize(pd->menu);
}

/********** Initialization & Finalization **********/
HopmanView*
hopman_view_init(XfcePanelPlugin *plugin)
{
    HopmanView   *view;                   /* internal use in this file */

    static int inotfd, sfd, mntfd;
    static GIOChannel *ginotfd, *gsfd, *gmntfd;
    static guint rc __attribute__((unused));

    DBG("initializing");
    g_assert(plugin != NULL);

    view            = g_new0(HopmanView, 1);
    view->plugin    = plugin;

    /* init button */

    DBG("init GUI");

    /* create the button */
    view->button = gtk_button_new ();
    GdkPixbuf *pPix = set_icon();
    view->image = gtk_image_new_from_pixbuf (pPix);
    gtk_widget_show(view->image);
    gtk_container_add (GTK_CONTAINER(view->button), view->image);
    gtk_button_set_relief (GTK_BUTTON(view->button), GTK_RELIEF_NONE);
    gtk_widget_set_tooltip_text( GTK_WIDGET(view->button), "Hopman");    
    xfce_panel_plugin_add_action_widget(view->plugin, view->button);
    gtk_container_add(GTK_CONTAINER(view->plugin), view->button);
    gtk_widget_show(view->button);

    /* button signal */
    g_signal_connect_swapped(view->button, "button-press-event",
                             G_CALLBACK(pview_cb_button_pressed), view);

    DBG("done");

    /*---------------- Record printstatus() as on_exit function ---------------*/
    on_exit(printstatus, NULL);

    /*---------------------- Native Language Translation ----------------------*/
    {
      int j;
      const char *var[3] = {"LC_MESSAGES", "LC_ALL", "LANG"};
      const char *value, *mylocale;
      for(j=0; j<3; j++)
        {
	      value=getenv(var[j]);
	      if(value) break;
        }
      mylocale = setlocale(LC_MESSAGES, value);
      fprintf( stderr, _("%s: language set to %s.\n"), progname, mylocale );
    }
    
    /*-------------------------------------------------------------------------*/
    /*          Obtain username and home, read configuration file,             */
    /*            kill previous instance and write pid file.                   */
    do_all_config();

    /*----------------- Decode paramaters for future use ----------------------*/
    showflags = config_show();
    decorate_window = config_bool(Decorate);
    toggle_hide     = config_bool(ToggleHide);
    keep_position   = config_bool(KeepPosition);
    {
      const char *text;
      text = config_string(InitialPosition);
      if(text) ipos = decode_pos(text); else ipos.ip_sty = None;
    }
   
    /*-------------------- Make the watch directory current -------------------*/
    if( chdir(watchdir) )
      {
        perror(watchdir);
        return NULL;
      }

    /*------ Initialize inotify & signalfd  and open /proc/self/mountinfo -----*/
    inotfd  = inotinit();
    ginotfd = g_io_channel_unix_new( inotfd );
    sfd     = signalinit();
    gsfd    = g_io_channel_unix_new( sfd );
    mntfd   = mountinit();
    gmntfd  = g_io_channel_unix_new( mntfd );
  
    /*----------------------- Declare input watches ---------------------------*/
    rc = g_io_add_watch(ginotfd, G_IO_IN,  callback_inotify, NULL);
    rc = g_io_add_watch(gsfd,    G_IO_IN,  callback_signal,  NULL);
    rc = g_io_add_watch(gmntfd,  G_IO_PRI, callback_mounts,  NULL);   

    return view;
}

void
hopman_view_finalize(HopmanView *view)
{
    pview_destroy_menu(view);

    if(view->button != NULL) {
        g_signal_handlers_disconnect_by_func(view->button,
                                             pview_destroy_menu,
                                             view);
        g_signal_handlers_disconnect_by_func(view->button,
                                             pview_cb_button_pressed,
                                             view);
        g_object_unref(view->button);
        view->button = NULL;
    }

    g_free(view);
}

// ----------------------------------------------------------------------
gboolean Get2Numbers (char *data) {
// ----------------------------------------------------------------------
    int n, i;

    n = strlen (data);
    if ((n == 0) | !isdigit (data[0]))
        return FALSE;
    menuX = atoi (data);
    i = 0;

    // Skip over first number
    while (isdigit (data[i])) {
        i++;
        if (i == n)
            return FALSE;
    }

    // Find start of the next number
    while (!isdigit (data[i])) {
        i++;
        if (i == n) return FALSE;
    }

    menuY = atoi (&data[i]);
    return TRUE;
}   // gboolean Get2Numbers

// ----------------------------------------------------------------------
int ReadLine () {
// ----------------------------------------------------------------------
    // Return kind of line, menu depth, and stripped text
    // return(-1) = Error, return(0) = EOF, return(>0) = keyword
    char *chop;
    int i, len, count, Kind;
    char tmp[MAX_LINE_LENGTH];
    char *str1, *str2;

    len = 0;
    while (len == 0) {
        // Read one line.
        if (fgets (Line, MAX_LINE_LENGTH, pFile) == NULL)
            return (0);
        strcpy (tmp, Line);
        lineNum++;

        // Remove comments
        chop = strchr (tmp, '#');
        if (chop != 0)
            *chop = '\0';

        len = strlen (tmp);

        // Remove trailing spaces & CR
        if (len > 0) {
            chop = &tmp[len - 1];
            while ((chop >= tmp) && (isspace (*chop) != 0)) {
                *chop = '\0';
                chop--;
            }
            len = strlen (tmp);
        }
    };

    // Big error?
    if (len >= MAX_LINE_LENGTH) {
        strncpy (data, tmp, MAX_LINE_LENGTH);
        data[MAX_LINE_LENGTH] = '\0';
        return (-1);
    }

    count = 0;

    // Calculate menu depth
    for (i = 0; i < len; i++) {
        if (tmp[i] == ' ')
            count += 1;
        else if (tmp[i] == '\t')    // Tab character = 4 spaces
            count += 4;
        else
            break;
    };
    depth = count / 4;

    // Remove leading white space
    if (count > 0) {
        str1 = tmp;
        str2 = tmp;
        while ((*str2 == ' ') || (*str2 == '\t')) {
            str2++;
            len--;
        }
        for (i = 0; i <= len; i++)
            *str1++ = *str2++;
    }

    if (strncasecmp (tmp, "separator", 9) == 0) {       // Found 'separator'
        strcpy (data, tmp);
        return (5);
    }
    else if (strchr (tmp, '=') == NULL) {               // Its a bad line
        strcpy (data, tmp);
        return (-1);
    }
    else if (strncasecmp (tmp, "iconsize", 8) == 0) {   // Found 'iconsize'
        Kind = 6;
    }
    else if (strncasecmp (tmp, "item", 4) == 0) {       // Found 'item'
        Kind = 1;
    }
    else if (strncasecmp (tmp, "cmd", 3) == 0) {        // Found 'cmd'
        Kind = 2;
    }
    else if (strncasecmp (tmp, "icon", 4) == 0) {       // Found 'icon'
        Kind = 3;
    }
    else if (strncasecmp (tmp, "submenu", 7) == 0) {    // Found 'submenu'
        Kind = 4;
    }
    else if (strncasecmp (tmp, "menupos", 7) == 0) {    // Found 'menupos'
        Kind = 7;
    }
    else if (strncasecmp (tmp, "fontsize", 8) == 0) {   // Found 'fontsize'
        Kind = 8;
    }
    else {                                              // Its a bad line
        strcpy (data, tmp);
        return (-1);
    }

    // Remove keywords and white space
    str2 = strchr (tmp, '=') + 1;
    while ((*str2 == ' ') || (*str2 == '\t'))
        str2++;
    strcpy (data, str2);


    return (Kind);
}   // int ReadLine

/*
// ----------------------------------------------------------------------
static void Expand_Tilde (char *txt, int maxLen) {
// ----------------------------------------------------------------------
    // Replace ~ in text

    static char home[MAX_HOME_LEN+1];
    static char buffer[BUFFER_LEN+1];
    static int  homeLen = 0; // Length of $HOME
    int len_txt;
    int i, j; // index to txt & buffer

    if (! strstr(txt, "~/")) return;

    if (maxLen > sizeof(buffer)) maxLen = sizeof(buffer);

    if (homeLen == 0) {
        strncpy (home, (char *) getenv("HOME"), MAX_HOME_LEN);
        homeLen = strlen(home);
        if (homeLen < strlen(getenv("HOME"))) {
            g_print("Error, length of $HOME > %d.\n", MAX_HOME_LEN);
            homeLen = 0;
            return;
        }
    }

    len_txt = strlen(txt);

    if (len_txt + homeLen >= maxLen) {
        g_print("Error, cannot expand '~', string is too long.\n");
        return;
    }

    i = j = 0;

    if ((txt[0] == '~') && (txt[1] == '/')) {
        strncpy(buffer, home, maxLen);
        i += 1;
        j += homeLen;
    } 
    else 
        buffer[j++] = txt[i++];
      
    while (i < len_txt) {
        if ((txt[i-1] == ' ') && (txt[i] == '~') && (txt[i+1] == '/')) {
            strncpy(&buffer[j], home, maxLen-j);
            i += 1;
            j += homeLen;
        }     
        else 
          buffer[j++] = txt[i++];

        if (j >= maxLen) {
            g_print("Error, cannot expand '~', string is too long.\n");
            return;
        }         
    }     

    strncpy(txt, buffer, maxLen);

return;
}
* */

/*============================ Print exit status ============================*/
void printstatus(int status, void *arg)
{
  switch(status)
    {
    case EXIT_SUCCESS:
      fprintf(stderr, _("%s terminated successfully.\n"), progname);
      break;
    case EXIT_FAILURE:
      fprintf(stderr, _("%s failed.\n"), progname);
      break;
    default:
      fprintf(stderr, _("%s terminated with status=%d\n"), progname, status);
    }
}

/*========================= Channel input callbacks =========================*/
gboolean callback_inotify( GIOChannel *source, GIOCondition condition,
			   gpointer pData )
{
  int inotfd;

  inotfd = (int)g_io_channel_unix_get_fd(source);
  inotify_read(inotfd);
  return TRUE;
}

gboolean callback_signal ( GIOChannel *source, GIOCondition condition,
			   gpointer pData )
{
  int sfd;

  sfd = (int)g_io_channel_unix_get_fd(source);
  signal_read( sfd );
  return TRUE;
}

gboolean callback_mounts ( GIOChannel *source, GIOCondition condition,
			   gpointer pData )
{
  mountpoints();
  return TRUE;
}

/*========================== Hide callback ===============================*/
void callback_hide( GtkWidget *widget, GdkEvent *event, gpointer *pData )
{
  
}

/*===== Interpret a character string defining a position on the screen ===== */
/* The position can be a pair of integers separated by white space or the    */
/* the single word "Mouse" (case independant).                               */
/* The integers' range is [0,100] .                                          */
/* All texts or values not matching these constraints will result in the     */
/* position being undefined.                                                 */

struct initial_position decode_pos(const char *text)
{
  struct initial_position ip;
  int n;
  //const char *style[3]={"None", "Mouse", "Percent"};
  ip.ip_sty = None;
  n = sscanf(text, " %d %d ", &ip.ip_x, &ip.ip_y);
  if(n==2)
    {
      if(ip.ip_x>=0 && ip.ip_x<=100 && ip.ip_y>=0 && ip.ip_y <=100)
	ip.ip_sty = Percent;
    }
  else if( !strcasecmp("mouse", text) ) ip.ip_sty = Mouse;

  return ip;
}
