/* See LICENSE file for copyright and license details. */
#include <sys/types.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <stdio.h>
#include <unistd.h>   // readlink()
#include <assert.h>
#include "hopman.h"
#include "watch.h"

/*=============================== WARNING ===================================*/
/* Beware: realloc() is used to increase/decrease the size of the memory     */
/* area used to store partition data, but realloc() may move this memory     */
/* area. Therefore pointers to partition's info are short-lived and cannot   */
/* be retained by the GUI, eg, for use by the callbacks.                     */
/*======================== THIS IS STILL TO BE DONE==========================*/
static partlist_t P = {0, NULL};
static int debug = -1;

// Usage: strncpy_t(str1, sizeof(str1), str2, strlen(str2));
// Copy string "in" with at most "insz" chars to buffer "out", which
// is "outsz" bytes long. The output is always 0-terminated. Unlike
// strncpy(), strncpy_t() does not zero fill remaining space in the
// output buffer:
static
char* strncpy_t(char* out, size_t outsz, const char* in, size_t insz)
{
   assert(outsz > 0);
   while(--outsz > 0 && insz > 0 && *in) { *out++ = *in++; insz--; }
   *out = 0;
   return out;
}

/*------------------------------- Creator -----------------------------------*/
void partition_new( ino_t inode, const char *name, const char *rootdev )
{
  char *buf = NULL;
  
  if(rootdev)
    {
       char path[NAME_MAX+1];
  
       strncpy_t(path, sizeof(path), "/sys/block/", strlen("/sys/block/"));
       strcat(path, rootdev);
       strcat(path, "/");
       strcat(path, name);
       strcat(path, "/partition");
  
       if (access(path, F_OK) == 0)
         return;
    }
  
  if(debug==-1)
    {
      debug = config_bool(Debug);
      if(debug == -1) debug = 0;
    }
  /* reallocate memory for the new array of partinfo_t pointers */
  P.p = realloc( P.p, sizeof(partinfo_t *) * (P.np+1) );

  if( P.np && !P.p )
    {
      fprintf( stderr, _("%s Cannot store partition data: %s\n"),
	       progname, strerror(errno) );
      exit( EXIT_FAILURE );
    }
  
  /* Allocate memory for this new partition */
  P.p[P.np] = malloc( sizeof(partinfo_t) );
  if(!P.p[P.np])
    {
      fprintf( stderr, _("%s Cannot store partition data: %s\n"),
	       progname, strerror(errno) );
      exit( EXIT_FAILURE );
    }
 
  /* set the inode and name of the new partition */
  P.p[P.np]->inode = inode;
  strncpy_t(P.p[P.np]->name, sizeof(P.p[P.np]->name), name, strlen(name));
  
  /* set the fstype and size of the new partition */  
  partition_get_fstype_from_blkid(name, &buf);
  strncpy_t(P.p[P.np]->fstype, sizeof(P.p[P.np]->fstype), buf, strlen(buf));
  free(buf);
  
  partition_get_size_from_lsblk(name, &buf);
  strncpy_t(P.p[P.np]->size, sizeof(P.p[P.np]->size), buf, strlen(buf));
  free(buf);
 
  P.p[P.np]->label[0] = P.p[P.np]->label[PM_LABEL_LEN] = '\0';
  P.p[P.np]->mnt[0] = P.p[P.np]->mnt[PM_MNT_LEN] = '\0';
  P.p[P.np]->nmounts = 0;

  /* Tell ui we have a new partition */
  if (strcmp(P.p[P.np]->fstype, "swap") && strncmp(P.p[P.np]->fstype, "--", 2))
     ui_create(P.p[P.np]);

  P.np ++;
  if (debug) partition_print();
  
  return;
}

/*------------------------------- Destructor --------------------------------*/
void partition_delete( const char *name )
{
  unsigned index;

  /* Find partition */
  for (index=0; index<P.np; index ++) if(!strcmp(P.p[index]->name, name)) break;
  if (index == P.np) return;

  /* Tell ui to remove partition from display */
  ui_delete(P.p[index]);

  /* free partition's data */
  free(P.p[index]);

  -- P.np;
  
  /* reshuffle the list */
  for( ; index < P.np; index ++) P.p[index] = P.p[index+1];
  
  /* reallocate memory for the new number of partitions */
  P.p = realloc( P.p, sizeof(partinfo_t *) * P.np );

  if( P.np && !P.p )
    {
      fprintf(stderr,_("%s Error freeing memory, returned by realloc(): %s\n"),
	       progname, strerror(errno) );
      exit( EXIT_FAILURE );
    }

  if(debug) partition_print();
  return;
}

/*------------------------------- Getters -----------------------------------*/
partlist_t partition_get_list(void)
{
  /* for use in mountpoints.c - dangerous, use with care ! */
  return P;
}

unsigned partition_by_name( const char *name )
{
  unsigned index;
  for(index=0; index<P.np; index++)
    {
      if( !strcmp(P.p[index]->name, name) ) break;
    }
  return index; /* If partition isn't found, return number of partitions */
}

unsigned partition_by_inode( ino_t inode )
{
  unsigned index;
  for(index=0; index<P.np; index++)
    {
      if( P.p[index]->inode == inode ) break;
    }
  return index; /* If partition isn't found, return number of partitions */
}

partinfo_t *get_partition( ino_t inode)
{
  int index;
  index = partition_by_inode(inode);
  return P.p[index];
}

/*--------------------------------- Setters ---------------------------------*/
void partition_set_label( ino_t inode, const char *label )
{
  unsigned index;
  int changed = 0;
  index = partition_by_inode(inode);
  if(index < P.np)
    {
      if( strcmp(P.p[index]->label, label) )
	{
	  strncpy(P.p[index]->label, label, PM_LABEL_LEN);
	  ui_update(P.p[index]);
	  changed ++;
	}
    }
  
  if(debug && changed) partition_print();
}

void partition_set_label_by_name( const char *name, const char *label )
{
  unsigned index;
  int changed = 0;
  index = partition_by_name(name);
  if(index < P.np)
    {
      if( strcmp(P.p[index]->label, label) )
	{
	  strncpy(P.p[index]->label, label, PM_LABEL_LEN);
	  ui_update(P.p[index]);
	  changed ++;
	}
    }
  
  if(debug && changed) partition_print();
}

void partition_set_label_by_blkid(const char *name)
{ 
    int rc;
    char *tmp = NULL;
    char cmd[256] = {0};
    char output[256];
    FILE *pf = NULL;
    char link[NAME_MAX+1];
  
    char *target = (char*) malloc(sizeof(char) * NAME_MAX + 1);
    if(! target) 
     {
        fprintf(stderr, _("%s: Memory allocation failure.\n"), progname);
        exit(EXIT_FAILURE);
     }
    
    if (! strncmp(name, "/dev/", strlen("/dev/")))
     {
        strncpy_t(link, sizeof(link), name, strlen(name));
     }
    else
     {
        strncpy_t(link, sizeof(link), "/dev/", strlen("/dev/"));
        strcat(link, name);
     }

    rc = readlink(link, target, NAME_MAX+1);    
    if (rc < 0)
    {
        fprintf(stderr, "[WARN]%s: readlink('%s') %s\n", 
                    progname, name, strerror(errno));
        free(target);    
        return;
    }
	
    tmp = rindex(target, '/') + 1;
    strncpy_t(cmd, sizeof(cmd), "/usr/sbin/usbmount get_label ", strlen("/usr/sbin/usbmount get_label "));  
    strcat(cmd, tmp);
  
    pf = popen(cmd, "r");
    if (! pf) {
        fprintf(stderr, "%s: fopen(%s): %s\n", progname, cmd, strerror(errno));
    }
  
    while(fgets(output, sizeof(output), pf)) {
        output[strcspn(output, "\n")] = '\0';
        partition_set_label_by_name(tmp, output);
  }
  
  pclose(pf);
  free(target);  
}

void partition_get_label_from_blkid(const char *name, char **buf)
{ 
    int rc;
    char *tmp = NULL;
    char cmd[256] = {0};
    char output[256];
    FILE *pf = NULL;
    char link[NAME_MAX + 1];
		
    *buf = (char*)malloc(sizeof(char) * PM_LABEL_LEN + 1);
    if (! *buf) {
        fprintf(stderr, "Memory allocation failure\n");
        exit (EXIT_FAILURE);
    }
  
    char *target = (char*)malloc(sizeof(char) * NAME_MAX + 1);
    if (! target) 
    {
        fprintf(stderr, _("%s: Memory allocation failure.\n"), progname);
        exit(EXIT_FAILURE);
    }
    
    if (! strncmp(name, "/dev/", strlen("/dev/")))
    {
        strncpy_t(link, sizeof(link), name, strlen(name));
    }
    else
    {
        strncpy_t(link, sizeof(link), "/dev/", strlen("/dev/"));
        strcat(link, name);
    }

    rc = readlink(link, target, NAME_MAX+1);    
    if(rc < 0)
    {
        fprintf(stderr, "[WARN]%s: readlink('%s') %s\n", 
                    progname, name, strerror(errno));
        free(target);    
        return;
    }
	
    tmp = rindex(target, '/') + 1;
    strncpy_t(cmd, sizeof(cmd), "/usr/sbin/usbmount get_label ", strlen("/usr/sbin/usbmount get_label "));  
    strcat(cmd, tmp);
  
    pf = popen(cmd, "r");
    if(!pf) {
        fprintf(stderr, "%s: fopen(%s): %s\n", progname, cmd, strerror(errno));
    }
  
    while(fgets(output, sizeof(output), pf)) {
        output[strcspn(output, "\n")] = '\0';
        strncpy_t(*buf, sizeof(*buf), output, strlen(output));
    }
  
    pclose(pf);
    free(target);  
}

void partition_get_fstype_from_blkid(const char *name, char **buf)
{ 
    int rc;
    char *tmp = NULL;
    char cmd[256] = {0};
    char output[256];
    FILE *pf = NULL;
    char link[1024];
		
    *buf = (char*)malloc(sizeof(char) * NAME_MAX + 1);
    if (! *buf) {
        fprintf(stderr, "Memory allocation failure\n");
        exit (EXIT_FAILURE);
    }

    strncpy_t(cmd, sizeof(cmd), "/usr/sbin/usbmount get_fstype ", strlen("/usr/sbin/usbmount get_fstype "));  
    strcat(cmd, name);
  
    pf = popen(cmd, "r");
    if(!pf) {
        fprintf(stderr, "%s: fopen(%s): %s\n", progname, cmd, strerror(errno));
    }
  
    while(fgets(output, sizeof(output), pf)) {
        output[strcspn(output, "\n")] = '\0';
        strncpy_t(*buf, sizeof(*buf), output, strlen(output));
    }
  
    pclose(pf);  
}

void partition_get_size_from_blkid(const char *name, char **buf)
{ 
  int rc;
  char cmd[256] = {0};
  char output[256];
  FILE *pf = NULL;
		
  *buf = (char*)malloc(sizeof(char) * PM_SIZE_LEN+1);
  if (!*buf) {
	fprintf(stderr, "Memory allocation failure\n");
	exit (EXIT_FAILURE);
  }
	
  strncpy_t(cmd, sizeof(cmd), "/usr/sbin/usbmount get_size ", strlen("/usr/sbin/usbmount get_size "));  
  strcat(cmd, name);

  pf = popen(cmd, "r");
  if(!pf) {
     fprintf(stderr, "%s: fopen(%s): %s\n", progname, cmd, strerror(errno));
  }
  
  while(fgets(output, sizeof(output), pf)) {
	 char *aux = NULL;
	 char *size = NULL;
     output[strcspn(output, "\n")] = '\0';
     if(strlen(output)<3) continue;
     aux = rindex( output, ':' ) + 3;
     aux[strcspn(aux, "\"")] = '\0';
     size = rindex(aux, '|') + 1;
     strncpy_t(*buf, sizeof(*buf), size, strlen(size));
  }
  
  pclose(pf);
}

void partition_get_size_from_lsblk(const char *name, char **buf)
{ 
  int rc;
  char cmd[512] = {0};
  char output[32] = {0};
  FILE *pf = NULL;
		
  *buf = (char*)malloc(sizeof(char) * PM_SIZE_LEN+1);
  if (!*buf) {
	fprintf(stderr, "Memory allocation failure\n");
	exit (EXIT_FAILURE);
  }
	
  strncpy_t(cmd, sizeof(cmd), "/bin/lsblk -o NAME,SIZE | grep ", strlen("/bin/lsblk -o NAME,SIZE | grep "));  
  strcat(cmd, name);

  pf = popen(cmd, "r");
  if(!pf) {
     fprintf(stderr, "%s: fopen(%s): %s\n", progname, cmd, strerror(errno));
  }
  
  while(fgets(output, sizeof(output), pf)) {
	 char *token = NULL; 
     output[strcspn(output, "\n")] = '\0';
     token = strtok(output, " ");
     token = strtok(NULL, " ");
     strncpy_t(*buf, sizeof(*buf), token, strlen(token));
  }
  
  pclose(pf);
}

/*-------------------------------- Debugger ---------------------------------*/
void partition_print(void)
{
  int i;
  if(P.np)
    fprintf(stderr,"   Inode Nm Device-Name    Label          Fstype        "
	    " Mount-Point\n");
  for( i=0; i<P.np; i++ )
    {
      fprintf(stderr, "%8ld %2d %-14s %-14s %-14s %-.20s\n",
	      P.p[i]->inode, P.p[i]->nmounts, P.p[i]->name, P.p[i]->label,
	      P.p[i]->fstype, P.p[i]->mnt);
    }
}

