#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <assert.h>
#include <sys/types.h>
#include <sys/wait.h>
#include "watch.h"


// Usage: strncpy_t(str1, sizeof(str1), str2, strlen(str2));
// Copy string "in" with at most "insz" chars to buffer "out", which
// is "outsz" bytes long. The output is always 0-terminated. Unlike
// strncpy(), strncpy_t() does not zero fill remaining space in the
// output buffer:
static char* strncpy_t(char* out, size_t outsz, const char* in, size_t insz)
{
   assert(outsz > 0);
   while(--outsz > 0 && insz > 0 && *in) { *out++ = *in++; insz--; }
   *out = 0;
   return out;
}

/*====== A subprogram like popen() but with diferences explained below ======*/
/* The pipe is to read in the current process and connected to stderr in the */
/* new one. The intent is to report error messages. Plus pid is returned.    */
/* If dirpath is an absolute path, chdir to it prior to excuting command.    */
FILE *epopen( const char *command,  pid_t *pid , const char *dirpath)
{
  int pipefd[2];
  pid_t p_id;
  FILE *pf;
  int rc;

  /*--------------------------- Create the pipe -----------------------------*/
  if(pipe(pipefd))
    {
      *pid = -1;
      return NULL;
    }

  /*------------------------------------ fork -------------------------------*/
  p_id = fork();
  switch(p_id)
    {
    case 0: /* the child */
      close(pipefd[0]);
      close(STDERR_FILENO);
      dup2(pipefd[1], STDERR_FILENO);
      close(pipefd[1]);
      if( *dirpath == '/' ) chdir( dirpath );
      /* restore original mask */
      if( sigprocmask(SIG_SETMASK, &oldset, NULL) )
	fprintf( stderr, _("%s: error detected by sigprocmask(): %s\n"),
		 progname, strerror(errno) );
      execl( "/bin/sh", "sh", "-c", command, (char *)NULL );
      fprintf( stderr, _("%s: error invoking /bin/sh: %s\n"),
	       progname, strerror(errno) );
      exit(EXIT_FAILURE); /* reminder: we are the child */
    case -1: /* fork() failed ! */
      rc = errno;
      close(pipefd[0]);
      close(pipefd[1]);
      errno = rc;
      *pid = -1;
      return NULL;
    default: /* the parent */
      close(pipefd[1]);
      pf = fdopen(pipefd[0], "r");
      *pid = p_id;
      return pf; /* The caller must close the pipe after use */
    }
}

int run_file_manager(const char *command, const char *dirpath)
{
	char cmd [512] = {0};
	
	strncpy_t(cmd, sizeof(cmd), command, strlen(command));

	if( *dirpath == '/' ) {
		chdir( dirpath );
		// In the case of SpaceFM, the above command that shoud change 
		// the current dir to the mountpoint doesn't take effect
		// So, additionally we'll pass it as an argument to the 
		// file manager
		strcat(cmd, " \"");
		strcat(cmd, dirpath);
		strcat(cmd, "\"");
	}
	return execl("/bin/sh", "sh", "-c", cmd, (char *)NULL );
}

int run_terminal_emulator(const char *command, const char *dirpath)
{
	if( *dirpath == '/' ) chdir( dirpath );
	return execl("/bin/sh", "sh", "-c", command, (char *)NULL );
}

int run_async_function( const char *command, const char *dirpath, int (*f)( const char*, const char* ) )
{
   int rc = 0;
   pid_t pid = 0;
   pid_t sid = 0;
   
   pid = fork();
   
   if( pid == 0 ) {  /* child */

      // detach from parent 
      sid = setsid();
      if( sid < 0 ) {
         fprintf(stderr, "%s: setsid failed: %s\n", progname, strerror(errno) );
         exit(1);
      }
      
      pid = fork();
      
      if( pid == 0 ) {   /* 2nd child */
         
         // run function
         return (*f)(command, dirpath);
      }
      else if( pid > 0 ) {
         
         exit(0);
      }
      else {

         fprintf(stderr, "%s: fork() failed: %s\n", progname, strerror(errno) );
         exit(-errno);
      }
   }
   else if( pid > 0 ) {
      
      rc = 0;
      
      // parent; succeeded
      // wait for intermediate process
      pid_t child_pid = waitpid( pid, &rc, 0 );
      if( child_pid < 0 ) {

         return -errno;
      }
      else if( WIFEXITED( rc ) && WEXITSTATUS( rc ) != 0 ) {
         
         rc = -WEXITSTATUS( rc );
      }
      
      return rc;
   }
   else {
      
      // error 
      rc = -errno;

      fprintf(stderr, "%s: fork() rc = %d\n", progname, rc);
      
      return rc;
   }
}



